/* FILE: mutils.c                               (D. Tottingham  03/24/91)

This is a collection of C utility functions that are used by xdetect and
supporting modules.  All functions have been written and compiled medium
model.  The following functions are included:

u_build_date ()                 build a date: YRMNDY
u_build_date_fn ()              build a date filename: pathname\YRMNDY
u_build_timeline ()             convert abs. time to an ascii string
u_convert_time ()               convert a struct timeb into abs. time
u_get_diskspace ()              get the percentage of free disk space
u_gmtime ()                     convert time value into structure.
u_ispow2 ()                     determine whether number is a power of two
u_log2 ()                       compute the base-2 logarithm of given number
u_strcpy ()                     copies far string2 to far string1
u_strncmp ()                    compare first n character of strings
u_strncpy ()                    copies n characters of far string2 to far string1
u_strnlen ()                    get length of far string
u_timestamp ()                  get timestamp
u_tzset_GMT ()                  set timezone to GMT

EXTERNAL FUNCTIONS CALLED:

er_abort ()                     display an error message then quit

HISTORY:

V1.95   (09/29/90)  Added functionality to u_ispow2() and u_log2() to make sure
                    that they only operate on positive, non-zero numbers.

                                                                         */

/*************************************************************************
                            INCLUDE FILES

*************************************************************************/
#include <dos.h>
#include <math.h>
#include <malloc.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys\types.h>
#include <sys\timeb.h>
#include <time.h>

#include "mconst.h"
#include "merror.h"
#include "mutils.h"


/*************************************************************************
                                GLOBALS

*************************************************************************/
int daylight;
long timezone;
char *tzname[];


/*=======================================================================*
 *                              u_build_date                             *
 *=======================================================================*/
/* Build a date: YRMNDY                                                  */

PUBLIC
long u_build_date (abs_time)
double abs_time;
{
   struct tm * local_tm;
   long date, ltime;

   ltime = (long) abs_time;
   local_tm = localtime (&ltime);

   date = (((long)local_tm->tm_year) * 10000) + ((local_tm->tm_mon + 1) * 100) +
          local_tm->tm_mday;

   return (date);
}

/*=======================================================================*
 *                            u_build_date_fn                            *
 *=======================================================================*/
/* Build a date filename: pathname\YRMNDY.                               */

PUBLIC
int u_build_date_fn ( abs_time, path_name, file_name )
double abs_time;
char path_name[];
char file_name[];
{
   struct tm * local_tm;
   long ltime;
   int j;

   ltime = (long) abs_time;
   local_tm = localtime (&ltime);

   file_name[0] = '\0';

   j = sprintf (file_name, "%s%02d%02d%02d", path_name,
                local_tm->tm_year,
                local_tm->tm_mon + 1,
                local_tm->tm_mday);

   return (j);
}

/*=======================================================================*
 *                            u_build_timeline                           *
 *=======================================================================*/
/* Convert the absolute time given in time_buf to an ascii string.       */

PUBLIC
char * u_build_timeline (abs_time, style)
double abs_time;
unsigned int style;
{
   char * timeline, * time_str;
   long time;
   unsigned int millitm;

   time = (long) abs_time;
   millitm = (unsigned int) ((abs_time - ((double) time)) * 1000);

   time_str = ctime (&time);
   timeline = calloc (( strlen(time_str) + MAX_NUMSTR_LENGTH), sizeof(char));

   if (timeline) {
      switch (style) {
         case 0:
            sprintf (timeline, "%.19s.%03d %s", time_str, millitm, &time_str[20]);
            break;
         case 1:
            sprintf (timeline, "%.19s.%03d %3s", time_str, millitm, tzname[0]);
            break;
         case 2:
            sprintf (timeline, "%.7s%.4s %.8s.%03d", &time_str[4], &time_str[20],
                     &time_str[11], millitm);
            break;
      }
      return (timeline);
   } else er_abort (U_NO_STORAGE);
}

/*=======================================================================*
 *                            u_convert_time                             *
 *=======================================================================*/
/* Convert struct timeb into an absolute time.                           */

PUBLIC
double u_convert_time (abs_timeb)
struct timeb abs_timeb;
{
   double abs_time;

   abs_time = (double) abs_timeb.time;
   abs_time += ((double) abs_timeb.millitm) / 1000.0;

   return (abs_time);
}

/*=======================================================================*
 *                             u_get_diskspace                           *
 *=======================================================================*/
/* Get the percentage of free disk space for this path.                  */

PUBLIC
long u_get_diskspace (path_name, drive_ptr)
char path_name[];
int * drive_ptr;
{
   static char * device_id[] = {NULL,"A:","B:","C:","D:","E:","F:","G:",NULL};
   struct diskfree_t diskspace;
   long percent_avail;
   int i;

   /* Look for drive letter in pathname */
   for (i = 1; device_id[i]; i++) {
      if (!strnicmp (device_id[i], path_name, 2))
         break;
   }

   /* If drive letter not found, get drive letter of cwd */
   if (!device_id[i]) _dos_getdrive (&i);

   /* Call _dos_getdiskfree to determine free disk space */
   if (!_dos_getdiskfree (i, &diskspace)) {
      percent_avail = ((long) diskspace.avail_clusters) * 100 /
                              diskspace.total_clusters;
      *drive_ptr = i - 1;
      return (percent_avail);
   }
   else return ( INVALID_DRIVE);
}

/*=======================================================================*
 *                                 u_gmtime                              *
 *=======================================================================*/
/* Convert time value into structure.                                    */

PUBLIC
void u_gmtime (time, tm_struct)
double time;
U_ELAPSED_TIME * tm_struct;
{
   unsigned long elapsed_time;

   elapsed_time = (unsigned long) time;

   tm_struct->millisec = (unsigned int) ((time - ((double) elapsed_time)) *
                         1000.0);

   tm_struct->days = elapsed_time / 86400;
   elapsed_time %= 86400;

   tm_struct->hours = elapsed_time / 3600;
   elapsed_time %= 3600;

   tm_struct->minutes = elapsed_time / 60;
   tm_struct->seconds = elapsed_time % 60;
}

/*=======================================================================*
 *                              u_ispow2                                 *
 *=======================================================================*/
/* Checks to see whether number is a power of two.                       */

PUBLIC
FLAG u_ispow2 (number)
unsigned long number;
{
   double powr;

   if (number == 0) return (FALSE);

   powr = u_log2 (number);
   if ( ceil(powr) > powr )
      return (FALSE);
   else
      return ( TRUE);
}

/*=======================================================================*
 *                               u_log2                                  *
 *=======================================================================*/
/* Compute the base-2 logarithm of given number.                         */

PUBLIC
double u_log2 (number)
unsigned long number;
{
   double ln_2, ln_d;

   if (number == 0) return (0.0);

   ln_d = log(((double) number));
   ln_2 = log(2.0);

   return (ln_d / ln_2);
}

/*=======================================================================*
 *                              u_strcpy                                 *
 *=======================================================================*/
/* Copies far string2 to far string1.                                    */

PUBLIC
char far * u_strcpy (string1, string2)
char far * string1;
char far * string2;
{
   int i;

   for (i = 0; i == 0 || string2[i-1] != '\0'; i++)
      string1[i] = string2[i];

   return (string2);
}

/*=======================================================================*
 *                              u_strncmp                                *
 *=======================================================================*/
/* Compare first n character of strings.                                 */

PUBLIC
int u_strncmp (string1, string2, n)
char far * string1;
char * string2;
unsigned int n;
{
   int i, length;

   if (u_strnlen (string1, n) != (length = strlen (string2))) return (0);
   for (i = 0; i < n && i <= length; i++)
      if (string1[i] > string2[i]) return (1);
      else if (string1[i] < string2[i]) return (-1);

   return (0);
}

/*=======================================================================*
 *                              u_strncpy                                *
 *=======================================================================*/
/* Copies n characters of far string2 to far string1.                    */

PUBLIC
char far * u_strncpy (string1, string2, n)
char far * string1;
char far * string2;
unsigned int n;
{
   int i;

   for (i = 0; i < n && (!i || string2[i-1] != '\0'); i++)
      string1[i] = string2[i];

   return (string1);
}

/*=======================================================================*
 *                             u_strnlen                                 *
 *=======================================================================*/
/* Get length of far string.                                             */

PUBLIC
unsigned int u_strnlen (string1, n)
char far * string1;
unsigned int n;
{
   int i;

   for (i = 0; i < n && string1[i] != '\0'; i++);

   return (i);
}

/*=======================================================================*
 *                            u_timestamp                                *
 *=======================================================================*/
/* Get timestamp.                                                        */

PUBLIC
double u_timestamp ()
{
   struct timeb current_timeb;

   ftime (&current_timeb);
   return ( u_convert_time (current_timeb));
}

/*=======================================================================*
 *                            u_tzset_GMT                                *
 *=======================================================================*/
/* Set timezone to GMT.                                                  */

PUBLIC
void u_tzset_GMT ()
{
   putenv ("TZ=GMT");
   tzset();
}
